;*******************************************************************************
; NES Joypad Adapter for Atari ProSytem & VCS                                  *
;*******************************************************************************
; Version 1.0, May 18th, 2015
; Copyright (C) 2015 Osman Celimli
;
; This software is provided 'as-is', without any express or implied
; warranty.  In no event will the authors be held liable for any damages
; arising from the use of this software.
;
; Permission is granted to anyone to use this software for any purpose,
; including commercial applications, and to alter it and redistribute it
; freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must not
;    claim that you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation would be
;    appreciated but is not required.
; 2. Altered source versions must be plainly marked as such, and must not be
;    misrepresented as being the original software.
; 3. This notice may not be removed or altered from any source distribution.
;-------------------------------------------------------------------------------
; Reads the serial input from a standard NES Joypad and converts the button
; scans into outputs which are compatile with either two-button ProSystem
; controllers or the ubiquitous one-button VCS joysticks.
;
; Two LED outputs (READYn and MODEn) let the user know whether or not the
; device is working and whether it is currently in ONE or TWO button mode.
;
; By default starts in two-button mode, but can be switched into one-button
; mode later on. Since the NES has a different (I would go as far to say
; "better") button layout and naming system, the following conversion
; is performed :
;
; Two-Button Mode
;----------------
;	A : RIGHT Trigger
;	B : LEFT Trigger
;	SELECT : Switch to one-button mode.
;	START : RIGHT + LEFT Triggers
;
; One-Button Mode
;----------------
;	A/B/START : Button (the only one)
;	SELECT: Switch to two-button mode.
;
; Designed for a 32.768KHz crystal since that is the cheapest frequency on the
; market. Of course you can clock things faster, 4MHz or more even. All that
; matters is that your clock isn't high enough to break the serial interface
; with the controllers.
;
; Note that while this source will quite happily run on any micro in the 16C5x
; or 16F5x family, the PICs in the 16F5x family will sometimes exhibit issues
; with LP (32.768KHz) oscillators.
;
	list		p=16c56a
	#include	<p16c56a.inc>

	__config _LP_OSC & _WDT_OFF & _CP_OFF

; Control Pins
;--------------
PA_LOAD		EQU 3
PA_CLK		EQU 2
PA_READYn	EQU 1
PA_DATA		EQU 0

PB_MODEn	EQU 3

; Joypad Scan Order
;-------------------
SCAN_A		EQU 0
SCAN_B		EQU 1
SCAN_SELECT	EQU 2
SCAN_START	EQU 3
SCAN_UP		EQU 4
SCAN_DOWN	EQU 5
SCAN_LEFT	EQU 6
SCAN_RIGHT	EQU 7

; Joypad State
;--------------
Joy_Vars	udata 0x08
NES_Cur		res 1
NES_Last	res 1
NES_Press	res 1
NES_Release	res 1

Atari_Mode	res 1
Atari_Cur	res 1

;*******************************************************************************
; Code Start                                                                   *
;*******************************************************************************
; Reset Vector @ $0000
; Note that the program counter actually defaults to all highs, and we just end
; up here as NOP = $000 and address wraparound.
Reset_Vec:	code 0x0000
	clrf	STATUS

	movlw	0x08
	movwf	FSR
Reset_ClearMem
	clrf	INDF		; $08-$1F is our RAM, a whole
	incf	FSR,f		; 24 Bytes of it. Zeroed here for
	btfss	FSR,5		; some semblance of convenience on
	goto	Reset_ClearMem	; an arch where there is none.

; Pin Configuration
;-------------------------------------------------------------------------------
; Port A
;	7-4: Not in x54,x56, Set as input in x55,x57
;	3: NES LOAD	(O), Latches Buttons when HIGH
;	2: NES CLK	(O)
;	1: READYn	(O)
;	0: NES DATA	(I)
	movlw	0xF1
	movwf	PORTA
	TRIS	PORTA
; Port B
;	7: Atari RIGHT	(O), Active LOW
;	6: Atari LEFT	(O), Active LOW
;	5: Atari UP	(O), Active LOW
;	4: Atari DOWN	(O), Active LOW
;	3: MODEn	(O), (LOW = ProSystem, HIGH = VCS)
;	2: Atari OLD	(O), Active LOW
;	1: Atari B	(O), Active HIGH
;	0: Atari A	(O), Active HIGH
	movlw	0xF4
	movwf	PORTB
	movlw	0x00
	TRIS	PORTB

	movlw	0x04
	OPTION

	movlw	0xFF
	movwf	NES_Last

;*******************************************************************************
; Main main main main... maine?                                                *
;*******************************************************************************
; Our main (what else?) loop. We'll be polling the controller and updating the
; button states sent back to the Atari as well as our internal modes at a
; minimum of ~60Hz. This should be fine.
main:
; The code below is about 64 instructions give or take pathing and errors in my
; counting. With a fixed 4 clocks per instruction, this means we're at
; (64 x 4) = 256 cycles or (256 / 32768Hz) = 128Hz.
;-------------------------------------------------------------------------------
; Now it's time for a fantastic journey to er... serially read in the state of
; the NES controller using the following procedure:
; 
; - Raise and lower LOAD to latch the state of the buttons.
; - The state of A can now be read over DATA
; - Raise and lower CLK. Once lowered the next button state can be read in
;   this order : B, SELECT, START, UP, DOWN, LEFT, RIGHT.
; - Repeat until all buttons have been read.
;-------------------------------------------------------------------------------
ScanJoy
	bsf	PORTA,PA_LOAD
	bcf	PORTA,PA_LOAD	; 2i

ScanJoy_Shift
	rrf	PORTA,w		; Unrolled since there's not much
	rrf	NES_Cur,f	; else that's going to use up
	bsf	PORTA,PA_CLK	; the program memory in this uC.
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f
	bsf	PORTA,PA_CLK
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f
	bsf	PORTA,PA_CLK
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f
	bsf	PORTA,PA_CLK
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f
	bsf	PORTA,PA_CLK
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f
	bsf	PORTA,PA_CLK
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f
	bsf	PORTA,PA_CLK
	bcf	PORTA,PA_CLK
	rrf	PORTA,w
	rrf	NES_Cur,f	; (4 x 7) + 2 = 30, 32i

	movf	NES_Cur,w
	xorwf	NES_Last,w
	andwf	NES_Cur,w
	movwf	NES_Release

	movf	NES_Cur,w
	xorwf	NES_Last,w	; Note that while the Cur and Last
	andwf	NES_Last,w	; readings are active low, the press
	movwf	NES_Press	; and release vars are active high.

	movf	NES_Cur,w
	movwf	NES_Last	; 42i

; Now that our new button state has been read in, perform our mode check and
; button conversion. Once everything has been prepped and stored in Atari_Cur,
; we'll flush the new button and mode state to PORTB.
;-------------------------------------------------------------------------------
	movlw	0xF4		; Default to nothing pressed
	movwf	Atari_Cur	; regardless of mode.
ModeCheck
	movlw	0xFF
	btfsc	NES_Press,SCAN_SELECT
	xorwf	Atari_Mode,f	; Switch modes on SELECT
ModeSet				; Mode Variable ($00 = 7800, $FF = 2600)
	btfsc	Atari_Mode,0	; MODEn Pin (Low = 7800, High = 2600)
	bsf	Atari_Cur,PB_MODEn

	btfsc	Atari_Mode,0
	goto	OneB_Start	; 51i

; Two-Button Conversion
;---------------------------------------
TwoB_Start
	movf	NES_Cur,w
	iorlw	0x0F		; Directions are all active-low so
	andwf	Atari_Cur,f	; we can just merge those in directly.

	comf	NES_Cur,w	; A and B need to be inverted prior to
	andlw	0x03		; addition since the 7800 buttons are
	iorwf	Atari_Cur,f	; active-high.

	movlw	0x03
	btfss	NES_Cur,SCAN_START
	iorwf	Atari_Cur,f	; Start = Both Left & Right Triggers
	goto	Atari_Write	; 61i

; One-Button Conversion
;---------------------------------------
OneB_Start
	movf	NES_Cur,w
	iorlw	0x0F
	andwf	Atari_Cur,f

	movlw	0xFB		; A, B, & Start all can set off the
	btfss	NES_Cur,SCAN_A	; jolly candylike button.
	andwf	Atari_Cur,f
	btfss	NES_Cur,SCAN_B
	andwf	Atari_Cur,f
	btfss	NES_Cur,SCAN_START
	andwf	Atari_Cur,f	; 61i

; PORTB Write
;-------------------------------------------------------------------------------
Atari_Write
	movf	Atari_Cur,w
	movwf	PORTB
	goto	main		; 64i

; MPASM seems to place a copy of the last issued instruction at the end of
; program memory with the END directive. Since this is also our reset vector in
; the PIC165x, I'm putting a NOP here explicitly so the PIC won't immediately
; jump to the main loop.
	nop
	END
