//------------------------------------------------------------------------------
// sublet.v
// Reduced-Size-and-Cost Memory Bastard for Atari 7800.
//------------------------------------------------------------------------------
// This mapper provides banking for 512KB of ROM, 32KB of RAM, optional graphic
// enhancement functionality, and control pins for interacting with an 8-Bit
// clocked output port implemeted on an external 74HC573.
//
// Compared to the normal SOUPER mapper, this reduced version loses the A8, A9,
// A10, and A11 passthroughs and requires the 8-Bit output port to be relocated
// to a 74HC573.
//
// The open-drain behavior of the audio command clock pin has also been removed,
// it now functions as a low/high toggler.
//
// Despite being "reduced," support for Flash ROM programming has been added
// through a few new registers at $808x and the romWrite_n pin. See below for
// programming details.
//------------------------------------------------------------------------------
// Version 1.0, March 13th, 2017
// Copyright (C) 2017 Osman Celimli
//
// This software is provided 'as-is', without any express or implied
// warranty.  In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would be
//    appreciated but is not required.
// 2. Altered source versions must be plainly marked as such, and must not be
//    misrepresented as being the original software.
// 3. This notice may not be removed or altered from any source distribution.
//------------------------------------------------------------------------------
module sublet(
	clk_phi2,
	reset_n,

	halt_n,
	data,
	rw,

	addr_15,
	addr_14,
	addr_13,
	addr_12,
	addr_7,
	addr_2,
	addr_1,
	addr_0,

	romSel_n,
	romWrite_n,
	ramSel_n,
	oe_n,
	wr_n,

	mapAddr,

	audCom,
	audReq_n
);
	// System Clocks and Reset
	//------------------------------
	input			clk_phi2;
	input			reset_n;

	// Bits of a Bus
	//------------------------------
	input			halt_n;
	input			rw;

	input[7:0]		data;
	input			addr_15;
	input			addr_14;
	input			addr_13;
	input			addr_12;
	input			addr_7;
	input			addr_2;
	input			addr_1;
	input			addr_0;

	// Memory Selects
	//------------------------------
	output			romSel_n;
	output			romWrite_n;
	output			ramSel_n;
	output			oe_n;
	output			wr_n;

	// Memory Banks (Connect to A7,A12+ on attached ROM + RAM)
	//------------------------------
	output[7:0]		mapAddr;

	// Audio Expansion Interface
	//------------------------------
	output			audCom;
	output			audReq_n;


//******************************************************************************
// !!!!----                BUSSING AND ADDRESS DECODING                 ----!!!!
//******************************************************************************
// As if the 6502 and 6800 weren't similar enough, rw and phi2 must be used to
// generate the usual oe_n and wr_n required for most memory.
//
// The 6502 will be READING while rw and phi2 are high, and WRITING if rw is
// low and phi2 is high. Basically, use PHI2 as the replacement for E compared
// to the 6800.
//
// Maria can take the bus whenever she pleases in order to fetch display lists
// and graphic data. Fortunately the 6502 is actually a special Atari variant
// (Sally) which can be single cycle halted + tri-stated with a special dance
// and Maria ONLY reads.
//
// My assumption is that Maria has the bus TWO falling edges of PHI2 after
// HALTn lowers. There shouldn't be any special requirements for completing
// multi-cycle RMW instructions ala stopping execution through the use of the
// RDYn pin.
//                        |
//                        V
//          __    __    __    __        __
// PHI2  __|  |__|  |__|  |__|  |__ ...   |___
//       ______                             __
// HALTn       |___________________ ... ___|
//
//------------------------------------------------------------------------------
	reg				haltDelA_ir,
					haltDelB_ir;
	wire			marRead_i;

assign oe_n = ~((clk_phi2 & rw) | marRead_i);
assign wr_n = ~(clk_phi2 & ~rw);

assign marRead_i = ~halt_n & haltDelB_ir;

always@(negedge clk_phi2 or negedge reset_n) begin
	if(~reset_n) begin
		haltDelA_ir <= 1'b0;
		haltDelB_ir <= 1'b0;
	end
	else if(~halt_n) begin
		haltDelA_ir <= ~halt_n;
		haltDelB_ir <= haltDelA_ir;
	end
	else begin
		haltDelA_ir <= 1'b0;
		haltDelB_ir <= 1'b0;
	end
end


// On reset, the 48KB area from $4000 - $FFFF available to cartridges is
// arranged as follows :
//
// $4000 - $7FFF : 16KB Extended RAM
// $8000 - $BFFF : 16KB Selectable ROM Bank
// $C000 - $FFFF : 16KB Fixed ROM Bank
//
// This is mostly compatible with the Atari SuperCart layout if RAM Banking
// is disabled. However, once RAM Banking is enabled by setting Souper Mode
// Bit 2 ($8003,2), the 16KB RAM region is repartitioned :
//
// $4000 - $5FFF : 8KB Fixed Extended RAM
// $6000 - $6FFF : 4KB Selectable V-Extended RAM
// $7000 - $7FFF : 4KB Selectable D-Extended RAM
//
// Enabling both SOUPER Mode and Character Remapping through Souper Mode
// Bits 0 & 1 ($8003,1 & 0), will allow additional Maria fetch trapping :
//
// - Fetches from $0000-$7FFF are unchanged
// - Fetches from $8000-$9FFF are routed to the Fixed ROM Bank
// - Fetches from $A000-$BFFF are routed to the Character A/B Bank Select
// - Fetches from $C000-$FFFF are routed to EXRAM
//------------------------------------------------------------------------------
	reg 			soupMode_ir;
	reg				flashLock_ir;

assign romSel_n = (marRead_i & soupMode_ir)
	? ~(addr_15 & ~addr_14)
	: ~(addr_15);
assign romWrite_n = ~((~flashLock_ir & ~wr_n) & (addr_15 & addr_14));

assign ramSel_n = (marRead_i & soupMode_ir)
	? ~(addr_14)
	: ~(~addr_15 & addr_14);


//******************************************************************************
// !!!!----               REGISTERS & EXPANSION INTERFACE               ----!!!!
//******************************************************************************
// A total of SIX mapping registers are available in the memory bastard, which
// are accessed by writing to $8000 - $807F and repeat over an 8-Byte range.
//
// Software should access these registers only using $8000 - $8007 in case newer
// variants of the mapper are developed with additional features.
//
// $0 = $8000 - $BFFF Bank Select, %xxxBBBBB
// $1 = Character A Graphic Select, %BBBBBBBS
// $2 = Character B Graphic Select, %BBBBBBBS
// $3 = Souper Mode Enable, %xxxxxECS
// $4 = $6000 - $6FFF EXRAM V-Bank Select, %xxxxxBBB
// $5 = $7000 - $7FFF EXRAM D-Bank Select, %xxxxxBBB
//------------------------------------------------------------------------------
	reg				chrMode_ir;
	reg				exMode_ir;

	reg[4:0]		bankSel_ir;
	reg[7:0]		chrSelA_ir,
					chrSelB_ir;
	reg[2:0]		exSelV_ir,
					exSelD_ir;

always@(negedge clk_phi2 or negedge reset_n) begin
	if(~reset_n) begin
		chrMode_ir <= 0;
		exMode_ir <= 0;
		soupMode_ir <= 0;

		chrSelA_ir <= 0;
		chrSelB_ir <= 0;
		exSelV_ir <= 0;
		exSelD_ir <= 0;
		bankSel_ir <= 0;
	end
	else if(~addr_7 & ~addr_14 & addr_15 & ~rw) begin
		case({addr_2, addr_1, addr_0})
			3'd0 : bankSel_ir <= data[4:0];
			3'd1 : chrSelA_ir <= data[7:0];
			3'd2 : chrSelB_ir <= data[7:0];
			3'd3 : begin
				soupMode_ir <= data[0];
				chrMode_ir <= data[1];
				exMode_ir <= data[2];
			end
			3'd4 : exSelV_ir <= data[2:0];
			3'd5 : exSelD_ir <= data[2:0];
		endcase
	end
end


// There is a SEVENTH register which is a special case, it is used to alter the
// state of the audio expansion communication port.
//
// Writing to $7 will alter invert audReq_n to let the audio processor know it
// has a command to read. Command data are latched in an external 74HC573 which
// is triggered by audCom rising on writes to $7.
//------------------------------------------------------------------------------
	reg				audReq_ir;

always@(negedge clk_phi2 or negedge reset_n) begin
	if(~reset_n) begin
		audReq_ir <= 1'b1;
	end
	else if(~addr_7 & ~addr_14 & addr_15 & ~rw) begin
		if(addr_2 & addr_1 & addr_0) begin
			audReq_ir <= ~audReq_ir;
		end
	end
end

assign audCom = (addr_15 & ~addr_14 & ~addr_7 & addr_2 & addr_1 & addr_0 & ~wr_n);
assign audReq_n = audReq_ir;


//******************************************************************************
// !!!!----            FLASH WRITE UNLOCK / PROTECT MECHANISM           ----!!!!
//******************************************************************************
// Upon reset the flash write line (romWrite_n) is stuck high, protecting the
// flash from erroneous modification by the game software. Unlocking flash
// writes and additional programming support are available through three new
// registers at $8080 - $8083 :
//
// $8080 = Flash Write Unlock, %KKKKKKKK
// $8081 = Flash Write Protect, %xxxxxxxx (any value)
// $8082 = $C000 - $FFFF Bank Select, %xxxBBBBB
//
// Writing the two ASCII characters 'WR' to $8080 will allow the flash to
// respond to writes ONLY in the $C000 - $FFFF range. This is normally the
// fixed bank which is mapped to the end of ROM, but can now be paged by
// writing to $8082 as a bank select.
//
// Once all desired modifications to the flash have been performed, it can
// be returned to a protected state by writing any value to $8081. This will
// also force the fixed bank to return to the last page of ROM and the game
// software may resume normal execution.
//
//
// Note that Maria MUST BE DISABLED before unlocking flash writes on a running
// console, otherwise her fetches may interfere with the programming sequence.
//------------------------------------------------------------------------------
	reg[4:0]		fixSel_ir;
	reg				keyState_ir;

always@(negedge clk_phi2 or negedge reset_n) begin
	if(~reset_n) begin
		flashLock_ir <= 1'b1;
		keyState_ir <= 1'b0;

		fixSel_ir <= 5'b11111;
	end
	else if(addr_7 & ~addr_14 & addr_15 & ~rw) begin
		case({addr_1, addr_0})
			2'd0 : begin
				// %00: FLASH WRITE UNLOCK (KEY == 'WR')
				//------------------------------
				if(flashLock_ir) begin
					if(~keyState_ir) begin
						// !!-- 'W' --!!
						keyState_ir <= (8'h57 == data)
							? 1'b1 : 1'b0;
						flashLock_ir <= 1'b1;
					end
					else begin
						// !!-- 'R' --!!
						keyState_ir <= (8'h52 == data)
							? 1'b1 : 1'b0;
						flashLock_ir <= (8'h52 == data)
							? 1'b0 : 1'b1;
					end
				end
			end
			2'd1 : begin
				// %01: FLASH WRITE PROTECT (ANY VALUE WILL LOCK)
				//------------------------------
				flashLock_ir <= 1'b1;
				keyState_ir <= 1'b0;

				fixSel_ir <= 5'b11111;
			end
			2'd2 : begin
				// %10: FIXED BANK SELECT
				//------------------------------
				if(~flashLock_ir) begin
					fixSel_ir <= data[4:0];
				end

				keyState_ir <= 1'b0;
			end
		endcase
	end
end


//******************************************************************************
// !!!!----                      ROM & RAM MAPPING                      ----!!!!
//******************************************************************************
assign mapAddr = ramSel_n
	// ---- ROM SELECT (DEFAULT) ----
	//------------------------------
	// If it was Maria and in $8000 - $9FFF, reroute the access to our FIXED
	// ROM BANK at $C000 - $FFFF which is the END of ROM (all address bits SET).
	//
	// If it was Maria and in $A000 - $BFFF, our address will be generated based
	// upon our character bank select register and Maria's current read address
	// in the format: %00000BBB BBBBHHHH SLLLLLLL. This effectively splits the
	// region into two 2KB graphic data viewports which can be anywhere in ROM.
	//
	//
	// If it was the 6502, $C000 - $FFFF are routed to the FIXED ROM BANK (last)
	// while $8000 - $BFFF are routed to the currently selected 16KB bank.
	//------------------------------
	? ((marRead_i & chrMode_ir)
		? (addr_13
			? (addr_7
				? {chrSelB_ir[7:1],
					chrSelB_ir[0]}
				: {chrSelA_ir[7:1],
					chrSelA_ir[0]})
			: {fixSel_ir, addr_13, addr_12,
				addr_7})


		: (addr_14
			? {fixSel_ir, addr_13, addr_12,
				addr_7}
			: {bankSel_ir, addr_13, addr_12,
				addr_7}))
	// ---- RAM SELECT ----
	//------------------------------
	// If it was Maria OR the 6502, see whether they were looking at the UPPER
	// or LOWER 8KB of the EXRAM region and if extended RAM banking is enabled.
	//
	// The LOWER 8KB is always fixed, but the two upper 4KB banks are
	// selectable in EXMODE.
	//------------------------------
	: ((addr_13 & exMode_ir)
		? (addr_12
			? {4'd0, exSelD_ir,
				addr_7}
			: {4'd0, exSelV_ir,
				addr_7})
		: {5'd0,
			addr_13, addr_12,
			addr_7});

endmodule

