;****************************************************************
; Speech Playback Routines					*
;****************************************************************
; Phrases are 8-Bit Unsigned Mono @ 16KHz and stored in an
; external SPI Flash ("phrase memory"). During playback, sample
; data are streamed in and then used to drive the state of PORTB
; which is connected to a DAC-08.
; The phrase memory is expected to be a 25FL216, but alternate
; flashes up to 16MB may be used as long as their read behavior
; is compatible. Data are stored on the flash using the MEMPAK
; format and assuming all file entries are phrase streams. The
; phrase memory should never be written to, only read. Just in
; case, write protect is pulled down on the phrase memory board.
;
; Starting from the base of the flash: 
;	$00-$05	"MEMPAK" In ASCII
;	$06-$07	Number of Files (Lo, Hi)
;	$08+	Directory entries in the following format:
;		$00-$01 ASCII File Type
;		$02-$04	Starting address of the File Data (Lo,Md,Hi)
;		$05-$07	Size of the File in Bytes (Lo,Md,Hi)
;	The remaining memory following the directory entries is
;	used for file data.
;
; When a phrase playback request is performed using Phrase_Say,
; Sylvie will check to see whether the phrase memory was verified
; to be okay at initialization time and if the phrase number
; is within the valid file number range. If either of these checks
; fails, then Sylvie will respond with '{SYL:??}'.
; Note that requesting a new phrase to be played will take priority
; over any currently playing phrases.
	list		p=16c73b
	#include	<p16c73b.inc>
	#include	"remote.inc"

; Equates
; (10MHz / 4 Presc) = 2.5MHz Base Clock
; (2.5MHz / 16KHz) = 156.25 Base Count
; (256 - 156) = 100, $64, Offset by a bit for latency
PHRASE_DIVIDER	EQU	0x76

; Vars
Phrase_Vars	udata
Phrase_Testing		res 1
Phrase_Playing		res 1
Phrase_CurSample	res 1
Phrase_ReqSample	res 1

Phrase_MemValid		res 1
Phrase_NumAvail		res 2 ; Lo,Hi

Phrase_StartAddr	res 3
Phrase_SamplesLeft	res 3 ; Lo,Md,Hi
	code
;****************************************************************
; Phrase_IRQ							*
;****************************************************************
Phrase_IRQ:
	global	Phrase_IRQ
	bcf	STATUS,RP0
; Note that since this interrupt will be turned on and off
; depending upon if the sample playback is enabled, we need
; to handle sample interrupts only if T0IE and T0IF are set.
	btfss	INTCON,T0IE
	return
	btfss	INTCON,T0IF
	return
; Clear the Timer 0 rollover flag and reload the up-counter.
	bcf	INTCON,T0IF
	movlw	PHRASE_DIVIDER
	movwf	TMR0
; Write the sample data to Port B immediately.
	movf	Phrase_CurSample,w
	movwf	PORTB
; If we're in testing mode just increment the value of CurSample
; to generate a sawtooth, then leave the handler.
	movf	Phrase_Testing,f
	btfsc	STATUS,Z
	goto	Phrase_CheckDone
	incf	Phrase_CurSample,f
	return
; If this was our last sample (current count is zero) then just
; disable the Timer 0 interrupt and clear the sample playing flag.
Phrase_CheckDone
	movf	Phrase_SamplesLeft + 0,w
	iorwf	Phrase_SamplesLeft + 1,w
	iorwf	Phrase_SamplesLeft + 2,w
	btfss	STATUS,Z
	goto	Phrase_SetupNext
	bcf	INTCON,T0IE
	clrf	Phrase_Playing
	goto	Phrase_CloseMemory

; Otherwise decrement our sample counter, and read
; the next byte of sample data from the flash.
Phrase_SetupNext
	decf	Phrase_SamplesLeft + 0,f
	incfsz	Phrase_SamplesLeft + 0,w
	goto	Phrase_FetchSample
	decf	Phrase_SamplesLeft + 1,f
	incfsz	Phrase_SamplesLeft + 1,w
	goto	Phrase_FetchSample
	decf	Phrase_SamplesLeft + 2,f
	incfsz	Phrase_SamplesLeft + 2,w
; Fall into Phrase_FetchSample from above...
;****************************************************************
; Phrase_FetchSample						*
; Read a single sample from the phrase memory and place it in	*
; Phrase_CurSample and W. Note that the Flash must have already	*
; been opened with Phrase_OpenMemory.				*
;****************************************************************
Phrase_FetchSample
	bcf	STATUS,RP0
	movlw	0x00
	movwf	SSPBUF
	bsf	STATUS,RP0
Phrase_FetchSample_Wait
	btfss	SSPSTAT,BF
	goto	Phrase_FetchSample_Wait

	bcf	STATUS,RP0
	movf	SSPBUF,w
	movwf	Phrase_CurSample
	return
;****************************************************************
; Phrase_OpenMemory						*
; Enable the MEMPAK and set its current read address to the	*
; contents of Phrase_StartAddr.					*
;****************************************************************
Phrase_OpenMemory
	bcf	STATUS,RP0
	bcf	PORTA,5
	movlw	0x03
	movwf	SSPBUF
	bsf	STATUS,RP0
Phrase_OpenMemory_WaitC
	btfss	SSPSTAT,BF
	goto	Phrase_OpenMemory_WaitC

	bcf	STATUS,RP0
	movf	Phrase_StartAddr + 2,w
	movwf	SSPBUF
	bsf	STATUS,RP0
Phrase_OpenMemory_WaitHi
	btfss	SSPSTAT,BF
	goto	Phrase_OpenMemory_WaitHi
	bcf	STATUS,RP0
	movf	Phrase_StartAddr + 1,w
	movwf	SSPBUF
	bsf	STATUS,RP0
Phrase_OpenMemory_WaitMd
	btfss	SSPSTAT,BF
	goto	Phrase_OpenMemory_WaitMd
	bcf	STATUS,RP0
	movf	Phrase_StartAddr + 0,w
	movwf	SSPBUF
	bsf	STATUS,RP0
Phrase_OpenMemory_WaitLo
	btfss	SSPSTAT,BF
	goto	Phrase_OpenMemory_WaitLo
	return
;****************************************************************
; Phrase_CloseMemory						*
; Release SPI_SEL to return the MEMPAK to an inactive state.	*
;****************************************************************
Phrase_CloseMemory
	bcf	STATUS,RP0
	bsf	PORTA,5
	return


;****************************************************************
; Phrase_ToggleTest						*
; Toggle whether the diagnostic sawtooth waveform generation	*
; is enabled or disabled.					*
;****************************************************************
Phrase_ToggleTest:
	global	Phrase_ToggleTest
	bcf	STATUS,RP0
	movf	Phrase_Playing,f
	btfss	STATUS,Z
	goto	Remote_ReqBusy

	comf	Phrase_Testing,f
	btfsc	STATUS,Z
	goto	Remote_ReqOkay

	clrf	Phrase_SamplesLeft + 0	; Preemptively clear the samples
	clrf	Phrase_SamplesLeft + 1	; left count, so that when the test
	clrf	Phrase_SamplesLeft + 2	; is disabled the IRQ will turn itself off.

	clrf	Phrase_CurSample	; The current sample var will be
	movlw	PHRASE_DIVIDER		; used as our current amplitude
	movwf	TMR0			; for the sawtooth, incremented
	bcf	INTCON,T0IF		; at the rate of the sample playback
	bsf	INTCON,T0IE		; interrupt (16KHz).
	goto	Remote_ReqOkay


;****************************************************************
; Phrase_Say							*
; Start playing the phrase number in W if it's within our valid	*
; phrase count and our phrase memory is OK. Otherwise respond	*
; with a '{SYL:??}'. Note that only the first 256 phrases in	*
; the MEMPAK may be played back using this subroutine.		*
;****************************************************************
Phrase_Say:
	global	Phrase_Say
	bcf	STATUS,RP0
	movwf	Phrase_ReqSample

	movf	Phrase_MemValid,f
	btfsc	STATUS,Z
	goto	Remote_ReqInvalid

	movf	Phrase_Testing,f
	btfss	STATUS,Z
	goto	Remote_ReqBusy

	movf	Phrase_NumAvail + 1,f
	btfss	STATUS,Z		; If we have over 255 files in
	goto	Phrase_Say_SelOK	; the MEMPAK, W must be fine.

	incf	Phrase_ReqSample,w	; But if we have less than that, make
	btfsc	STATUS,Z		; sure our target phrase is in range.
	goto	Remote_ReqInvalid
	subwf	Phrase_NumAvail + 0,w
	btfss	STATUS,C
	goto	Remote_ReqInvalid
Phrase_Say_SelOK
	movf	Phrase_ReqSample,w
	movwf	Phrase_CurSample
	movlw	0xFF
	movwf	Phrase_Playing
	
	call	Phrase_CloseMemory	; Ensure the MEMPAK is released
	clrf	Phrase_StartAddr + 0	; before starting our new read stream.
	clrf	Phrase_StartAddr + 1
	clrf	Phrase_StartAddr + 2

	movf	Phrase_CurSample,w	; Base address of the file's directory
	addlw	0x01			; is 8 + 8 * FileNumber
	rlf	Phrase_StartAddr + 1,f
	movwf	Phrase_StartAddr + 0
	rlf	Phrase_StartAddr + 0,f
	rlf	Phrase_StartAddr + 1,f
	rlf	Phrase_StartAddr + 0,f
	rlf	Phrase_StartAddr + 1,f
	rlf	Phrase_StartAddr + 0,f
	rlf	Phrase_StartAddr + 1,f	; But we'll skip the first two bytes since
	bsf	Phrase_StartAddr + 0,1	; they're only used for the file type.

	call	Phrase_OpenMemory	; Open the MEMPAK at our file's directory
	call	Phrase_FetchSample	; entry, then read in our starting offst...
	movwf	Phrase_StartAddr + 0
	call	Phrase_FetchSample
	movwf	Phrase_StartAddr + 1
	call	Phrase_FetchSample
	movwf	Phrase_StartAddr + 2

	call	Phrase_FetchSample	; ...and file length.
	movwf	Phrase_SamplesLeft + 0
	call	Phrase_FetchSample
	movwf	Phrase_SamplesLeft + 1
	call	Phrase_FetchSample
	movwf	Phrase_SamplesLeft + 2

	call	Phrase_CloseMemory	; Then close and reopen the MEMPAK at the
	call	Phrase_OpenMemory	; base address of our phrase file and
	call	Phrase_FetchSample	; read the first byte.

	movlw	PHRASE_DIVIDER		; We can now start our 16KHz interrupt
	movwf	TMR0			; using Timer 0.
	bcf	INTCON,T0IF
	bsf	INTCON,T0IE
	goto	Remote_ReqOkay


;****************************************************************
; Phrase_Init							*
;****************************************************************
Phrase_Init:
	global	Phrase_Init
; Reset our memory valid flag and number of phrases available
; count, we'll only set them if the phrase memory is verified okay.
	bcf	STATUS,RP0
	clrf	Phrase_MemValid
	clrf	Phrase_NumAvail + 0
	clrf	Phrase_NumAvail + 1
	clrf	Phrase_Testing
	clrf	Phrase_Playing
; Our 8-Bit DAC is connected to Port B, center it.
	movlw	0x80
	movwf	PORTB
; Make sure the MEMPAK is released before proceeding (A5 High) and
; the MISO, MOSI and CLK lines are already set in their desired orientation.
	bsf	PORTA,5
	bsf	STATUS,RP0
	bcf	TRISA,5
	movlw	0xD1
	movwf	TRISC
; Set the SSP into SPI Master / Mode 3 and to perform transactions
; at 2.5MHz (10MHz / 4). Then set all of our pin directions early.
; We can now perform full-duplex transactions with the MEMPAK by
; lowering A5 and writing to SSPBUF.
	bcf	STATUS,RP0
	movlw	0x30
	movwf	SSPCON
; Our 16KHz IRQ can be generated through Timer 0 using the base
; clock of 2.5MHz (10MHz / 4) and a divider value of 156.
; Note that Timer 0 uses an UP counter, so this value must
; be complemented (interrupt occurs on $FF -> $00), 0x63.
	bsf	STATUS,RP0
	bcf	OPTION_REG,T0CS	; <- Nice naming guys, really...
	bsf	OPTION_REG,PSA
; We can now check the integrity of the MEMPAK, open it and scan
; addresses $0-$5 for "MEMPAK" then copy the file count from $6-$7.
	bcf	STATUS,RP0
	clrf	Phrase_StartAddr + 0
	clrf	Phrase_StartAddr + 1
	clrf	Phrase_StartAddr + 2
	call	Phrase_OpenMemory

	call	Phrase_FetchSample
	sublw	'M'
	btfss	STATUS,Z
	goto	Phrase_CloseMemory
	call	Phrase_FetchSample
	sublw	'E'
	btfss	STATUS,Z
	goto	Phrase_CloseMemory
	call	Phrase_FetchSample
	sublw	'M'
	btfss	STATUS,Z
	goto	Phrase_CloseMemory
	call	Phrase_FetchSample
	sublw	'P'
	btfss	STATUS,Z
	goto	Phrase_CloseMemory
	call	Phrase_FetchSample
	sublw	'A'
	btfss	STATUS,Z
	goto	Phrase_CloseMemory
	call	Phrase_FetchSample
	sublw	'K'
	btfss	STATUS,Z
	goto	Phrase_CloseMemory

; MEMPAK seems to be okay, so let's copy our file count and
; set the phrase memory valid flag.
	call	Phrase_FetchSample
	movwf	Phrase_NumAvail + 0
	call	Phrase_FetchSample
	movwf	Phrase_NumAvail + 1
	comf	Phrase_MemValid,f
	goto	Phrase_CloseMemory
	END
