;****************************************************************
; Remote Control Interface (Using USART)			*
;****************************************************************
; Sylvie must receive commands in the following format: {SYLxYY}
; Where x is the command type, and YY is the one byte of command
; data in uppercase textual hexadecimal (for example 'EF').
;
; The following commands are supported:
; N-- : Null-Acknowledge.
;	Sylvie just responds with '{SYL:Hi}'
; EBB : Set Eye Brightness.
;	BB may range from 00-FF and is used to set Sylvie's eye
;	brightness from lowest to highest respectively.
; VSS : Say Phrase.
;	SS will be used as the phrase which Sylvie will say.
;	Out of range phrase numbers will yield a '??' response.
; D-- : Toggle DAC Test
;	Enable or disable the generation of a sawtooth wave
;	over the DAC used for phrase playback. Note that while
;	the DAC test is enabled, Sylvie cannot perform phrase playback.
;
; Sylvie responds in a similar 8-Character arrangement: {SYL:YY}
; Where YY is an ASCII response. Sylvie is usually quiet but
; will give the following responses:
; 'Up'	Broadcast at power-up.
; '--'	Broadcast repeatedly during attract mode.
; 'Hi'	Response to a Null-Acknowledge.
; 'OK'	Response to a properly executed command.
; '..'	Response to an ignored command (too busy)
; '??'	Response to an invalid command or out of range parameters.
; 'XX'	Response to a command with bad parameter formatting.
	list		p=16c73b
	#include	<p16c73b.inc>
	#include	"eyes.inc"
	#include	"phrase.inc"

; Equates
RESPONSE_HI	EQU	0x01
RESPONSE_OK	EQU	0x02
RESPONSE_BUSY	EQU	0x03
RESPONSE_FAULT	EQU	0x04
RESPONSE_INVAL	EQU	0x05

; Vars
Remote_Vars	udata
Remote_RxBuf	res 8
Remote_RxChar	res 1
Remote_RxOff	res 1

Remote_TxBuf	res 16
Remote_TxChar	res 1
Remote_TxIn	res 1
Remote_TxOut	res 1

Remote_HexVal	res 1
Remote_Command	res 1
Remote_Param	res 1

Remote_GotCmd	res 1
	global	Remote_GotCmd
Remote_RespReq	res 1
	code
;****************************************************************
; Remote_IRQ							*
; IRQ Handler for Tx and Rx events. Will check to see if either	*
; condition occured and should just be called from IRQ_Vec.	*
;****************************************************************
Remote_IRQ:
	global Remote_IRQ
;------------------------
; First check on receives
Remote_IRQ_CheckRx
; We need to check for a framing error first, these can
; be resolved simply by reading the receive register again.
	bcf	STATUS,RP0
	btfsc	RCSTA,FERR
	movf	RCREG,w
; If we have valid data to investigate, RCIF should
; still be set.
	btfss	PIR1,RCIF
	goto	Remote_IRQ_CheckOver
	movf	RCREG,w
	call	Remote_HandleRx
; If an overrun error has occured, we need to reset the receive
; hardware by disabling it, then enabling it again
Remote_IRQ_CheckOver
	bcf	STATUS,RP0
	btfss	RCSTA,OERR
	goto	Remote_IRQ_CheckTx
	bcf	RCSTA,CREN
	nop
	bsf	RCSTA,CREN
	nop
	
;-----------------------
; Now check on transmits
Remote_IRQ_CheckTx
	bcf	STATUS,RP0
	btfss	PIR1,TXIF
	return
; If the transmit ready flag is set, we're ready to send our
; next character. However, if no data remains to be sent (ringbuffer
; offsets are equal), then we can just disable the transmit IRQ.
	movf	Remote_TxIn,w
	subwf	Remote_TxOut,w
	btfss	STATUS,Z
	goto	Remote_IRQ_TxNotDone

	bsf	STATUS,RP0
	bcf	PIE1,TXIE
	return
Remote_IRQ_TxNotDone
; Send the next byte and increment our ringbuffer outgoing offset.
	movf	Remote_TxOut,w
	addlw	Remote_TxBuf
	movwf	FSR
	movf	INDF,w
	movwf	TXREG

	movf	Remote_TxOut,w
	addlw	1
	andlw	0xF
	movwf	Remote_TxOut
Remote_IRQ_Done
	return


;****************************************************************
; Remote_HandleRx						*
; Handle the character just received (must be loaded into W),	*
; taking appropriate action if a full command has been issued.	*
;****************************************************************
Remote_HandleRx:
	bcf	STATUS,RP0
	movwf	Remote_RxChar
; Add the character to the receive ringbuffer and then
; increment our current receive offset.
	movf	Remote_RxOff,w
	addlw	Remote_RxBuf
	movwf	FSR
	movf	Remote_RxChar,w
	movwf	INDF

	movf	Remote_RxOff,w
	addlw	1
	andlw	0x7
	movwf	Remote_RxOff
; If the last character received is a right curlybrace '}' then
; attempt to decode the last seven characters as a command.
	movf	Remote_RxChar,w
	sublw	'}'
	btfss	STATUS,Z
	return
; We might have gotten valid data, check for '{SYL'
	movlw	0
	call	Remote_FetchRxChar
	sublw	'{'
	btfss	STATUS,Z
	return
	movlw	1
	call	Remote_FetchRxChar
	sublw	'S'
	btfss	STATUS,Z
	return
	movlw	2
	call	Remote_FetchRxChar
	sublw	'Y'
	btfss	STATUS,Z
	return
	movlw	3
	call	Remote_FetchRxChar
	sublw	'L'
	btfss	STATUS,Z
	return
; Looks like we got a packet in the format of '{SYLxYY}' but we
; still need to verify and dispatch the command x and parameter YY.
; First of all, pull them out of the ringbuffer and verify the
; parameter is in proper ASCII-Hex ('00'-'FF') format.
	movlw	4
	call	Remote_FetchRxChar
	movwf	Remote_Command

	movlw	5
	call	Remote_FetchRxChar
	call	Remote_CharToHex
	btfsc	STATUS,C
	goto	Remote_ReqFault
	movwf	Remote_Param
	movlw	6
	call	Remote_FetchRxChar
	call	Remote_CharToHex
	btfsc	STATUS,C
	goto	Remote_ReqFault
	swapf	Remote_Param,f
	iorwf	Remote_Param,f
; If an error was encountered while attempting to pre-parse the
; command and parameters, send out a fault notifier '{SYL:XX}'
; otherwise fall into Remote_DoCommand.


;****************************************************************
; Remote_DoCommand						*
; Check and dispatch the command and parameters stored in	*
; Remote_Command and Remote_Param.				*
;****************************************************************
Remote_DoCommand:
; Not sure if it's valid yet...
; But hey, we got at least one command! We can kick out of
; attract mode if we're still there.
	movlw	0xFF
	movwf	Remote_GotCmd

; N-- : Null-Acknowledge.
Remote_CheckNull
	movf	Remote_Command,w
	sublw	'N'
	btfss	STATUS,Z
	goto	Remote_CheckEyes
	goto	Remote_ReqHi

; EBB : Set Eye Brightness.
Remote_CheckEyes
	movf	Remote_Command,w
	sublw	'E'
	btfss	STATUS,Z
	goto	Remote_CheckDAC
	movf	Remote_Param,w
	call	Eyes_SetIntensity
	goto	Remote_ReqOkay

; D-- : Toggle DAC Test
Remote_CheckDAC
	movf	Remote_Command,w
	sublw	'D'
	btfss	STATUS,Z
	goto	Remote_CheckPhrase
	goto	Phrase_ToggleTest

; VSS : Say Phrase.
Remote_CheckPhrase
	movf	Remote_Command,w
	sublw	'V'
	btfss	STATUS,Z
	goto	Remote_ReqInvalid
	movf	Remote_Param,w		; We've run out of possible commands,
	goto	Phrase_Say		; send an invalid notifier '{SYL:??}'
;****************************************************************
; Remote_CharToHex						*
; Attempt to convert the character stored in W to hexadecimal	*
; (source is expected to be '0-9' or 'A-F'). If the source char	*
; is outside of the expected range, the carry flag will be set.	*
;****************************************************************
Remote_CharToHex:
	movwf	Remote_HexVal
	btfss	Remote_HexVal,7		; Must be < 0x80 (Traditional ASCII Range)
	goto	Remote_CharToHex_Num
Remote_CharToHex_Fail
	bsf	STATUS,C
	return

Remote_CharToHex_Num
	sublw	'9'			; 0-9 -> 0x30-0x39
	btfss	STATUS,C
	goto	Remote_CharToHex_AF	; Greater than 0x39, check for A-F

	movf	Remote_HexVal,w
	addlw	0xD0
	btfss	STATUS,C
	goto	Remote_CharToHex_Fail	; Less than 0x30, invalid
	movwf	Remote_HexVal
	bcf	STATUS,C
	return

Remote_CharToHex_AF
	movf	Remote_HexVal,w		; A-F -> 0x41-0x46
	sublw	'F'
	btfss	STATUS,C
	goto	Remote_CharToHex_Fail	; Greater than 0x46, invalid

	movf	Remote_HexVal,w
	addlw	0xBF
	btfss	STATUS,C
	goto	Remote_CharToHex_Fail	; Between 0x39 and 0x41, invalid
	addlw	0x0A
	movwf	Remote_HexVal
	bcf	STATUS,C
	return
;****************************************************************
; Remote_FetchRxChar						*
; Fetch the character at the specified index (w) in the recieve	*
; ringbuffer relative to the current recieve offset.		*
;****************************************************************
Remote_FetchRxChar:
	bcf	STATUS,RP0
	addwf	Remote_RxOff,w
	andlw	0x7
	addlw	Remote_RxBuf
	movwf	FSR
	movf	INDF,w
	return




;****************************************************************
; Remote_HandleReq						*
; Background subroutine for handling response requests made	*
; from our interrupt handler. Call during idle time / main.	*
;****************************************************************
Remote_HandleReq:
	global Remote_HandleReq
	bcf	STATUS,RP0
	movlw	HIGH(Remote_SendResp_Table)
	movwf	PCLATH
	movlw	LOW(Remote_SendResp_Table)
	addwf	Remote_RespReq,w
	clrf	Remote_RespReq
	btfsc	STATUS,C
	incf	PCLATH,f
	movwf	PCL

Remote_SendResp_Table
	return				; Nothing
	goto	Remote_SendHi		; RESPONSE_HI, '{SYL:Hi}'
	goto	Remote_SendOkay		; RESPONSE_OK, '{SYL:OK}'
	goto	Remote_SendBusy		; RESPONSE_BUSY, '{SYL:..}'
	goto	Remote_SendFault	; RESPONSE_FAULT, '{SYL:XX}'
	goto	Remote_SendInvalid	; RESPONSE_INVAL, '{SYL:??}'
	return				; Wait, what?
;****************************************************************
; Response Requests						*
; Since Remote_SendChar is not IRQ-Safe (it can get stuck	*
; waiting for the transmit ringbuffer to empty) we can only	*
; make transfer requests in IRQs and then perform the actual	*
; enqueues from the main loop. Use these to request a response	*
; in any function which needs to send data over serial and	*
; will be executing in an IRQ.					*
;****************************************************************
; '{SYL:Hi}'
Remote_ReqHi:
	global Remote_ReqHi
	bcf	STATUS,RP0
	movlw	RESPONSE_HI
	movwf	Remote_RespReq
	return
; '{SYL:OK}'
Remote_ReqOkay:
	global Remote_ReqOkay
	bcf	STATUS,RP0
	movlw	RESPONSE_OK
	movwf	Remote_RespReq
	return
; '{SYL:..}'
Remote_ReqBusy:
	global Remote_ReqBusy
	bcf	STATUS,RP0
	movlw	RESPONSE_BUSY
	movwf	Remote_RespReq
	return
; '{SYL:XX}'
Remote_ReqFault:
	global Remote_ReqFault
	bcf	STATUS,RP0
	movlw	RESPONSE_FAULT
	movwf	Remote_RespReq
	return
; '{SYL:??}'
Remote_ReqInvalid:
	global Remote_ReqInvalid
	bcf	STATUS,RP0
	movlw	RESPONSE_INVAL
	movwf	Remote_RespReq
	return




;****************************************************************
; Remote_SendChar						*
; Add the character in W to the outgoing transmit queue,	*
; enabling the transmit interrupt if needed. Should not be used	*
; in an IRQ Handler. Make a new entry above for IRQ responses.	*
;****************************************************************
Remote_SendChar:
	global Remote_SendChar
	bcf	STATUS,RP0
	movwf	Remote_TxChar

SendChar_WaitEmpty
	movf	Remote_TxIn,w
	addlw	1
	andlw	0xF
	subwf	Remote_TxOut,w
	btfsc	STATUS,Z		; Wait for the ringbuffer
	goto	SendChar_WaitEmpty	; to empty if it is full...

	movf	Remote_TxIn,w
	addlw	Remote_TxBuf
	movwf	FSR
	movf	Remote_TxChar,w		; Then shove the supplied
	movwf	INDF			; character into it.

; We can then adjust the incoming offset and enable transmit IRQs.
	movf	Remote_TxIn,w
	addlw	1
	andlw	0xF
	movwf	Remote_TxIn
	bsf	STATUS,RP0
	bsf	PIE1,TXIE
	return
;****************************************************************
; Remote_SendPreface						*
; Add '{SYL:' to the outgoing transmit ringbuffer.		*
;****************************************************************
Remote_SendPreface:
	global Remote_SendPreface
	movlw	'{'
	call	Remote_SendChar
	movlw	'S'
	call	Remote_SendChar
	movlw	'Y'
	call	Remote_SendChar
	movlw	'L'
	call	Remote_SendChar
	movlw	':'
	goto	Remote_SendChar
;****************************************************************
; Remote_SendHi							*
; Send out a friendly greeting '{SYL:Hi}'			*
;****************************************************************
Remote_SendHi:
	global Remote_SendHi
	call	Remote_SendPreface
	movlw	'H'
	call	Remote_SendChar
	movlw	'i'
	call	Remote_SendChar
	movlw	'}'
	goto	Remote_SendChar
;****************************************************************
; Remote_SendOkay						*
; Send out a success notifier '{SYL:OK}'			*
;****************************************************************
Remote_SendOkay:
	global Remote_SendOkay
	call	Remote_SendPreface
	movlw	'O'
	call	Remote_SendChar
	movlw	'K'
	call	Remote_SendChar
	movlw	'}'
	goto	Remote_SendChar
;****************************************************************
; Remote_SendBusy						*
; Send out a busy notifier '{SYL:..}'				*
;****************************************************************
Remote_SendBusy:
	global Remote_SendBusy
	call	Remote_SendPreface
	movlw	'.'
	call	Remote_SendChar
	movlw	'.'
	call	Remote_SendChar
	movlw	'}'
	goto	Remote_SendChar
;****************************************************************
; Remote_SendFault						*
; Send out a fault notifier '{SYL:XX}'				*
;****************************************************************
Remote_SendFault:
	global Remote_SendFault
	call	Remote_SendPreface
	movlw	'X'
	call	Remote_SendChar
	movlw	'X'
	call	Remote_SendChar
	movlw	'}'
	goto	Remote_SendChar
;****************************************************************
; Remote_SendInvalid						*
; Send out an invalid command/parameters notifier '{SYL:??}'	*
;****************************************************************
Remote_SendInvalid:
	global Remote_SendInvalid
	call	Remote_SendPreface
	movlw	'?'
	call	Remote_SendChar
	movlw	'?'
	call	Remote_SendChar
	movlw	'}'
	goto	Remote_SendChar




;****************************************************************
; Remote_Init							*
; Initialize the Tx/Rx buffers and configure the USART for	*
; asynchronous transmit and receive in 8N1 format @ 2400bps.	*
;****************************************************************
Remote_Init:
	global Remote_Init
; Before doing anything, reset our incoming and outgoing data offsets.
	bcf	STATUS,RP0
	clrf	Remote_RxOff
	clrf	Remote_TxIn
	clrf	Remote_TxOut
; No commands in yet either...
	clrf	Remote_GotCmd
	clrf	Remote_RespReq
; Since we have a system clock of 10MHz, we can approximate some
; common baud rates using the following parameters:
;  Fast Prescale: 10MHz / 16 = 625KHz Base Tick, Div 64 (+1) -> 9615bps
;  Slow Prescale: 10MHz / 64 = 156.25KHz Base Tick, Div 64 (+1) -> 2403bps
;  Slow Prescale: 10MHz / 64 = 156.25KHz Base Tick, Div 129 (+1) -> 1201bps
; Note the fast prescaler is slightly unstable in the 16C73 and 16C73A
; and may cause corruption, this has been corrected in the 16C73B.
	bsf	STATUS,RP0
; For 9615bps
;	bsf	TXSTA,BRGH
;	movlw	0x40
; For 2403bps
	bcf	TXSTA,BRGH
	movlw	0x40
; For 1201bps
;	bcf	TXSTA,BRGH
;	movlw	0x81
	movwf	SPBRG
; Enable the serial port on RC7/Rx and RC6/Rx as well
; as transmits and receives.
	bsf	TXSTA,TXEN
	bcf	STATUS,RP0
	bsf	RCSTA,SPEN
	bsf	RCSTA,CREN
; We can now enable receive interrupts, although they won't
; be serviced until peripheral interrupts are enabled later on...
	bsf	STATUS,RP0
	bsf	PIE1,RCIE
	return
	END
